// Resolve browser inconsistences when clicking on buttons
document.addEventListener('click', event => {
  if (event.target.matches('button')) {
    event.target.focus()
  }
})

// ========================
// Variables
// ========================
const tabby = document.querySelector('.tabby')
const tabsList = tabby.querySelector('.tabs')
const tabs = [...tabby.querySelectorAll('.tab')]
const tabContents = [...tabby.querySelectorAll('.tab-content')]

// ========================
// Functions
// ========================
/**
 * Selects a tab and its corresponding tab-content
 * @param {HTMLElement} tab The tab to select
 */
const selectTab = tab => {
  const target = tab.dataset.target
  const tabContent = tabby.querySelector('#' + target)

  // Selects a tab
  tabs.forEach(t => {
    t.classList.remove('is-selected')
    t.removeAttribute('aria-selected')
    t.setAttribute('tabindex', '-1')
  })
  tab.classList.add('is-selected')
  tab.setAttribute('aria-selected', 'true')
  tab.removeAttribute('tabindex')

  // Selects the corresponding tab content
  tabContents.forEach(c => c.classList.remove('is-selected'))
  tabContent.classList.add('is-selected')
}

/**
 * Gets the previous tab
 * @param {Number} index Index of a tab
 * @returns {HTMLElement} The previous tab
 */
const getPreviousTab = index => {
  if (index !== 0) {
    return tabs[index - 1]
  }
}

/**
 * Gets the next tab
 * @param {Number} index Index of a tab
 * @returns {HTMLElement} The next tab
 */
const getNextTab = index => {
  if (index !== tabs.length - 1) {
    return tabs[index + 1]
  }
}

// ========================
// Execution
// ========================
// Selects clicked tab
tabsList.addEventListener('click', event => {
  const tab = event.target
  selectTab(tab)
})

// Select previous or next tab when user press Left or Right arrow keys
tabsList.addEventListener('keydown', event => {
  const { key } = event
  if (key !== 'ArrowLeft' && key !== 'ArrowRight') return

  const index = tabs.findIndex(t => t.classList.contains('is-selected'))

  let targetTab
  if (key === 'ArrowLeft') targetTab = getPreviousTab(index)
  if (key === 'ArrowRight') targetTab = getNextTab(index)

  if (targetTab) {
    event.preventDefault()
    targetTab.click()
  }
})

// Shifts focus from tab to tabpanel
tabsList.addEventListener('keydown', event => {
  const key = event.key
  if (event.shiftKey) return
  if (key !== 'Tab' && key !== 'ArrowDown') return

  const tab = event.target
  const target = tab.dataset.target
  const tabPanel = tabby.querySelector('#' + target)

  event.preventDefault()
  tabPanel.focus()
})
