const defaults = {
  buttonElement: '',
  modalElement: '',
  afterOpen () {},
  type: 'normal',
  delayBeforeOpening: 0
}

export default function Modal (settings) {
  settings = Object.assign({}, defaults, settings)

  const { type } = settings
  switch (type) {
    case 'normal': UserTriggeredModal(settings); break
    case 'timed': TimedModal(settings); break
  }
}

function BaseModal (settings) {
  const { modalElement, buttonElement } = settings
  const overlayElement = modalElement.parentElement
  const contentElement = modalElement.querySelector('.modal__content')
  const closeButtonElement = modalElement.querySelector('.jsModalClose')

  function trapFocus (event) {
    const focusables = getKeyboardFocusableElements(modalElement)
    const firstFocusable = focusables[0]
    const lastFocusable = focusables[focusables.length - 1]

    // Directs to first focusable
    if (document.activeElement === lastFocusable && event.key === 'Tab' && !event.shiftKey) {
      event.preventDefault()
      firstFocusable.focus()
    }

    // Directs to last focusable
    if (document.activeElement === firstFocusable && event.key === 'Tab' && event.shiftKey) {
      event.preventDefault()
      lastFocusable.focus()
    }
  }

  const modal = {
    get isOpen () {
      return overlayElement.classList.contains('is-open')
    },

    get siblingElements () {
      return [...overlayElement.parentElement.children]
        .filter(el => el !== overlayElement)
    },

    showSiblingElements () {
      modal.siblingElements.forEach(element => {
        element.removeAttribute('aria-hidden')
      })
    },

    hideSiblingElements () {
      modal.siblingElements.forEach(element => {
        element.setAttribute('aria-hidden', true)
      })
    },

    open () {
      overlayElement.classList.add('is-open')
      document.addEventListener('keydown', trapFocus)
      modal.hideSiblingElements()

      // Autofocus
      const focusableEls = getKeyboardFocusableElements(contentElement)
      if (focusableEls[0]) focusableEls[0].focus()

      settings.afterOpen()
    },

    close () {
      overlayElement.classList.remove('is-open')
      document.removeEventListener('keydown', trapFocus)
      buttonElement.setAttribute('aria-expanded', false)
      buttonElement.focus()
      modal.showSiblingElements()
    }
  }

  closeButtonElement.addEventListener('click', event => {
    modal.close()
  })

  overlayElement.addEventListener('click', event => {
    if (!event.target.closest('.modal')) {
      modal.close()
    }
  })

  document.addEventListener('keydown', event => {
    if (modal.isOpen && event.key === 'Escape') {
      modal.close()
    }
  })

  return modal
}

function UserTriggeredModal (settings) {
  const modal = BaseModal(settings)
  const { buttonElement } = settings

  buttonElement.addEventListener('click', event => {
    modal.open()
    buttonElement.setAttribute('aria-expanded', true)
  })
}

function TimedModal (settings) {
  const modal = BaseModal(settings)
  setTimeout(_ => modal.open(), settings.delayBeforeOpening)
}

function getKeyboardFocusableElements (element = document) {
  return [...element.querySelectorAll(
    'a, button, input, textarea, select, details, [tabindex]'
  )]
    .filter(el => !el.hasAttribute('disabled'))
    .filter(el => !el.hasAttribute('tabindex') || el.getAttribute('tabindex') >= 0)
}
