/* globals getComputedStyle */
export default function Carousel (carouselElement) {
  const contents = carouselElement.querySelector('.carousel__contents')
  const slides = [...carouselElement.querySelectorAll('.carousel__slide')]
  const previousButton = carouselElement.querySelector('.previous-button')
  const nextButton = carouselElement.querySelector('.next-button')
  const liveregion = carouselElement.querySelector('[role="status"]')
  const dotsContainer = createDots(slides)
  const dots = [...dotsContainer.children]

  const carousel = {
    get currentSlideIndex () {
      const currentSlide = contents.querySelector('.is-selected')
      return slides.findIndex(slide => slide === currentSlide)
    },

    /**
     * Switches slides
     * @param {number} currentSlideIndex
     * @param {number} targetSlideIndex
     */
    switchSlide (currentSlideIndex, targetSlideIndex) {
      const currentSlide = slides[currentSlideIndex]
      const targetSlide = slides[targetSlideIndex]

      // Switches to the correct slide
      const destination = getComputedStyle(targetSlide).left
      contents.style.transform = `translateX(-${destination})`
      currentSlide.classList.remove('is-selected')
      targetSlide.classList.add('is-selected')

      // Highlights the correct dot
      const currentDot = dots[currentSlideIndex]
      const targetDot = dots[targetSlideIndex]
      currentDot.classList.remove('is-selected')
      targetDot.classList.add('is-selected')

      // Show/hide arrows
      if (targetSlideIndex === 0) {
        previousButton.setAttribute('hidden', true)
        nextButton.removeAttribute('hidden')
      } else if (targetSlideIndex === dots.length - 1) {
        previousButton.removeAttribute('hidden')
        nextButton.setAttribute('hidden', true)
      } else {
        previousButton.removeAttribute('hidden')
        nextButton.removeAttribute('hidden')
      }

      // Roving tabindex
      const currentLink = slides[currentSlideIndex].querySelector('a')
      const targetLink = slides[targetSlideIndex].querySelector('a')
      currentLink.setAttribute('tabindex', '-1')
      targetLink.removeAttribute('tabindex')

      // Announce selected slide to screen reader
      liveregion.textContent = `Slide ${targetSlideIndex + 1} of ${slides.length} selected`
    },

    // Listeners
    showNextSlide () {
      const currentSlideIndex = carousel.currentSlideIndex
      const nextSlideIndex = currentSlideIndex + 1
      carousel.switchSlide(currentSlideIndex, nextSlideIndex)
    },

    showPreviousSlide () {
      const currentSlideIndex = carousel.currentSlideIndex
      const previousSlideIndex = currentSlideIndex - 1
      carousel.switchSlide(currentSlideIndex, previousSlideIndex)
    },

    handleClicksOnDots (event) {
      const dot = event.target.closest('button')
      if (!dot) return

      const currentSlideIndex = carousel.currentSlideIndex
      const targetSlideIndex = dots.findIndex(d => d === dot)
      carousel.switchSlide(currentSlideIndex, targetSlideIndex)
    },

    handleKeydown (event) {
      const { key } = event
      if (key !== 'ArrowLeft' && key !== 'ArrowRight') return

      const currentSlideIndex = carousel.currentSlideIndex
      let targetSlideIndex

      if (key === 'ArrowLeft') targetSlideIndex = currentSlideIndex - 1
      if (key === 'ArrowRight') targetSlideIndex = currentSlideIndex + 1

      if (targetSlideIndex < 0) targetSlideIndex = 0
      if (targetSlideIndex > slides.length - 1) targetSlideIndex = slides.length - 1

      carousel.switchSlide(currentSlideIndex, targetSlideIndex)

      // Focus on the correct slide
      const targetLink = slides[targetSlideIndex].querySelector('a')
      targetLink.focus({ preventScroll: true })

      // Fallback for preventScroll
      setTimeout(() => {
        contents.parentElement.scrollLeft = 0
      }, 0)
    }
  }

  carouselElement.appendChild(dotsContainer)
  setSlidePositions(slides)

  nextButton.addEventListener('click', carousel.showNextSlide)
  previousButton.addEventListener('click', carousel.showPreviousSlide)
  dotsContainer.addEventListener('click', carousel.handleClicksOnDots)
  contents.addEventListener('keydown', carousel.handleKeydown)
}

/**
 * Creates dots for the carousel
 * @returns The HTML for dots
 */
function createDots (slides) {
  const dotsContainer = document.createElement('div')
  dotsContainer.classList.add('carousel__dots')
  dotsContainer.setAttribute('aria-hidden', true)

  slides.forEach(slide => {
    const dot = document.createElement('button')
    dot.classList.add('carousel__dot')
    dot.setAttribute('tabindex', -1)

    if (slide.classList.contains('is-selected')) {
      dot.classList.add('is-selected')
    }

    dotsContainer.appendChild(dot)
  })
  return dotsContainer
}

/**
 * Sets slide positions
 */
function setSlidePositions (slides) {
  const slideWidth = slides[0].getBoundingClientRect().width
  slides.forEach((slide, index) => {
    slide.style.left = slideWidth * index + 'px'
  })
}
